<?php
use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;
use WC_Payment_Gateway;

class WC_teip_Payment_Gateway extends WC_Payment_Gateway{

    public $plugin_version;
    public $test_mode;
    public $url;
    public $uid;
    public $email;
    public $password;

    public function __construct() {

        $this->plugin_version = '1.0';  
        $this->id = 'teip_gateway';
        $this->method_title = __('Teip Gateway', 'woocommerce');
        $this->method_description = __('Accept payments through Teip Gateway.', 'woocommerce');
        $this->icon = 'https://assets.teip.io/S3/static/logo-red.svg'; // payment gateway icon
        $this->has_fields = true;

        $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');

        $this->enabled = $this->get_option('enabled');
        $this->title = $this->get_option('title');

        // Load the init form fields
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->enabled = $this->get_option('enabled');
        $this->test_mode = $this->get_option('test_mode');
        $this->uid = $this->get_option('uid');
        $this->email = $this->get_option('email');
        $this->password = $this->get_option('password');

        if ($this->test_mode == 'yes') {
            $this->url = 'https://sandbox.teip.io';
        } else {
            $this->url = 'https://api.teip.io';
        }

        $this->supports = array( 'products' );

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
              'title'   => __('Enable/Disable', 'teip-gateway'),
              'type'    => 'checkbox',
              'label'   => __('Enable Teip Gateway', 'teip-gateway'),
              'default' => 'yes',
            ),
            'title' => array(
              'title'       => 'Título',
              'type'        => 'text',
              'description' => __('Ingrese el título que desea mostrar durante el pago.','teip-payment-gateway'),
              'default'     => __('Pasarela de Pago Teip','teip-payment-gateway'),
              'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __('Descripción','teip-payment-gateway'),
                'type'        => 'textarea',
                'description' => __('Ingrese la descripción que desea mostrar durante el pago.','teip-payment-gateway'),
                'default'     => __('Paga con tu pasarela de pago Teip.','teip-payment-gateway'),
                'desc_tip'    => true,
            ),
            'test_mode' => array(
                'title'       => __('Habilitar modo de prueba','teip-payment-gateway'),
                'label'       => __('Activar / Desactivar','teip-payment-gateway'),
                'type'        => 'checkbox',
                'description' => __('Al habilitarlo confirma que el UID, Email and Password correspondan al ambiente de pruebas.','teip-payment-gateway'),
                'desc_tip'     => true
            ),
            'uid' => array(
                'title'       => 'UID',
                'type'        => 'text',
                'description' => __('Ingrese el UID (client_id) proporcionado por Teip.','teip-payment-gateway'),
                'desc_tip'    => true,
            ),
            'email' => array(
                'title'       => 'Email',
                'type'        => 'email',
                'description' => __('Ingrese el correo electrónico proporcionado por Teip.','teip-payment-gateway'),
                'desc_tip'    => true,
            ),
            'password' => array(
                'title'       => 'Password',
                'type'        => 'text',
                'description' => __('Ingrese la contraseña proporcionado por Teip.','teip-payment-gateway'),
                'desc_tip'    => true,
            ),
        );
    }

    public function payment_fields() {
        error_log("render payment_fields");
        echo "
        <fieldset id='wc-".esc_attr( $this->id )."-cc-form' class='wc-credit-card-form wc-payment-form' style='background:transparent;'>
            <!-- Renderizar los campos del formulario de tarjeta de crédito aquí -->
            <input id='teip_ccno' name='ccno' type='text' autocomplete='off' maxlength='16' placeholder='0000 0000 0000 0000'>
            <input id='teip_mmexpdate' name='mmExpDate' type='text' autocomplete='off' maxlength='2' placeholder='MM'>
            <input id='teip_yyexpdate' name='yyExpDate' type='text' autocomplete='off' maxlength='2' placeholder='YY'>
            <input id='teip_cvc' name='cvc' type='password' autocomplete='off' maxlength='3' placeholder='CVC'>
        </fieldset>";
    }

    public function process_payment($order_id) {
        global $woocommerce;

        // get order detailes
        $order = wc_get_order( $order_id );

        $authorize_code = $this->teip_api_authorize();
        $authorize_access_token = $this->teip_api_authorize_token($authorize_code);
        $authentication_access_token = $this->teip_api_authentication($authorize_access_token);
        $pay_data = $this->teip_api_pay($authentication_access_token, $order);

        if (!is_wp_error( $pay_data)) {
            // it could be different depending on your payment processor
            if ($pay_data == 'success') {

                // we received the payment
                $order->payment_complete();
                //$order->reduce_order_stock();

                // notes to customer
                $order->add_order_note( __('Su pagó se realizo con éxito! Su orden ha sido completada. Gracias!', 'teip-payment-gateway'), true );
                $order->add_order_note( __('Esta nota privada se muestra solo en la página de edición del pedido', 'teip-payment-gateway'), false );

                // empty cart
                $woocommerce->cart->empty_cart();

                // redirect to the thank you page
                return array(
                    'result' => 'success',
                    'redirect' => $this->get_return_url($order)
                );

            } else {
                wc_add_notice(__('Error A05 - No se pudo completar el proceso de pago. Por favor intente de nuevo.','teip-payment-gateway'), 'error');
                return;
            }
        } else {
            wc_add_notice(__('Error W01 - Error de conexión.','teip-payment-gateway'), 'error');
            return;
        }
    }
    
    public function teip_api_authorize() {

        // Teip Authorize
        $endpoint = $this->url . '/api/v1/oauth/application/authorize';
        error_log($endpoint);

        $code_challenge = '2QQd8XiagyNpPFraq4nHTPxyK5LaQMPTef6C3hIwFtU'; //$this->uid;

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $endpoint);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            'Accept: application/json',
            'Content-Type: application/json',
            'WP-Teip-Payment-Gateway-Plugin-Version: ' . $this->plugin_version
        ));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        $data = <<<DATA
        {
            "client_id": "$this->uid",
            "redirect_uri": "urn:ietf:wg:oauth:2.0:oob",
            "response_type": "code",
            "code_challenge": "$code_challenge",
            "code_challenge_method": "S256"
        }
        DATA;

        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);

        $resp = curl_exec($curl);
        curl_close($curl);
        error_log($resp);
        $result = json_decode($resp);

        if ($result->status == 'redirect') {
            error_log("authorize ok");
            return $result->redirect_uri->code;
        }

        return wc_add_notice(  __('Error A01 - No se pudo crear el código. Por favor intente de nuevo. Si el inconveneniente persiste por favor comuniquese con el equipo de soporte.','teip-payment-gateway') , 'error' );
    }

    public function teip_api_authorize_token($authorize_code) {
        // Teip Authorize Token
        $endpoint = $this->url . '/api/v1/oauth/application/token';
        $verifier = 'TQRCt5K-9diIPVE5H3lDmWt3TeLwqbZr5Cm5mDixyhnuNuebHgmU0nkfZl8io07lOowcD1JdEFcOopABwISLcQ';

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $endpoint);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            'Accept: application/json',
            'Content-Type: application/json',
            'WP-Teip-Payment-Gateway-Plugin-Version: ' . $this->plugin_version
        ));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        error_log($authorize_code);
        error_log($this->uid);

        $data = <<<DATA
        {
            "client_id": "$this->uid",
            "code": "$authorize_code",
            "code_verifier": "$verifier",
            "redirect_uri": "urn:ietf:wg:oauth:2.0:oob",
            "grant_type": "authorization_code"
        }
        DATA;

        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);

        $resp = curl_exec($curl);
        curl_close($curl);
        error_log($resp);
        $result = json_decode($resp);
        

        if ($result->access_token) {
            error_log("authorize token ok");
            return  $result->access_token;
        }

        return wc_add_notice(__('Error A02 - No se pudo obtener el token. Por favor intente de nuevo. Si el inconveneniente persiste por favor comuniquese con el equipo de soporte.','teip-payment-gateway'), 'error');
    }


    public function teip_api_authentication($authorize_access_token) {
        error_log($authorize_access_token);
        // Teip Authentication Token
        $endpoint = $this->url . '/api/v1/oauth/token';

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $endpoint);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            "Authorization: Bearer ". $authorize_access_token,
            'Accept: application/json',
            'Content-Type: application/json',
            'WP-Teip-Payment-Gateway-Plugin-Version: ' . $this->plugin_version
        ));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        $data = <<<DATA
        {
            "grant_type" : "password",
            "email" : "$this->email",
            "password" : "$this->password",
            "scope": "business"
        }
        DATA;

        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);

        $resp = curl_exec($curl);
        curl_close($curl);
        error_log($resp);

        $result = json_decode($resp);

        if ($result->access_token) {
            error_log("api auth access token ok");
            return  $result->access_token;
        }

        return wc_add_notice(__('Error A03 - No se pudo autenticar la petición. Por favor intente de nuevo. Si el inconveneniente persiste por favor comuniquese con el equipo de soporte.','teip-payment-gateway'), 'error');
    }

    public function teip_api_pay($authentication_access_token, $order) {
        error_log($authentication_access_token);
        // Teip Pay
        $endpoint = $this->url . '/api/v1/pay';

        // Order field values
        $fullname = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        $email = $order->get_billing_email();
        $total = strval($order->get_total() * 100);
        $country  = $order->get_billing_country();
        $city  = $order->get_billing_city();
        $address = $order->get_billing_address_1() . ' ' . $order->get_billing_address_2();
        $postcode = $order->get_billing_postcode();
        $orderID = $order->get_id();

        // Payment field values
        $ccno = $_POST[ 'ccno' ];
        $mmExpDate = $_POST[ 'mmExpDate' ];
        $yyExpDate = $_POST[ 'yyExpDate' ];
        $cvc = $_POST[ 'cvc' ];

        // Confirm card network
        if ($ccno[0] ==  4) {
            $card_network = "visa";
        }  elseif ($ccno[0] ==  5) {
            $card_network = "mastercard";
        } else {
            return  wc_add_notice(__('Error P01 - Número de tarjeta no valida. Por favor intente de nuevo con una tarjet Visa o Mastercard.','teip-payment-gateway'), 'error');
        }

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $endpoint);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            "Authorization: Bearer ". $authentication_access_token,
            'Accept: application/json',
            'Content-Type: application/json',
            'WP-Teip-Payment-Gateway-Plugin-Version: ' . $this->plugin_version
        ));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        $data = <<<DATA
        {
            "data": {
            "card_network": "$card_network",
            "card_number": "$ccno",
            "security_code": "$cvc",
            "cardholder_name": "$fullname",
            "expiration_month": "$mmExpDate",
            "expiration_year": "$yyExpDate",
            "subtotal_cents": "$total",
            "subtotal_currency": "USD",
            "billing_email": "$email",
            "billing_name": "$fullname",
            "billing_country": "$country",
            "billing_city": "$city",
            "billing_address": "$address",
            "billing_postal_code": "$postcode",
            "reference_code": "$orderID"
            }
        }
        DATA;

        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);

        $resp = curl_exec($curl);
        curl_close($curl);

        $result = json_decode($resp);

        if ($result->data) {
            error_log("pay ok");
            return  $result->data->status;
        }

        return wc_add_notice(__('Error A04 - No se  pudo procesar el pago. Por favor intente de nuevo. Si el inconveneniente persiste por favor comuniquese con el equipo de soporte.','teip-payment-gateway'), 'error');
    }


    public function is_available() {
        if ($this->enabled === 'yes') {
            return true;
        }
    
        // Añadir registro de depuración
        if (defined('WP_DEBUG') && WP_DEBUG) {
            wc_get_logger()->error('teip Gateway is not available: ' . $this->id);
        }
        
        return false;
    }
    
}

class WC_Gateway_Teip_Block extends AbstractPaymentMethodType {
    public function initialize() {
        // Inicializa tu método de pago aquí
    }

    public function get_name() {
        return 'teip_gateway';
    }

    public function get_payment_method_script_handles() {
        return ['teip-block-script'];
    }

    public function is_active() {
        return true;
    }

    public function get_payment_method_data() {
        return [
            'title' => __('Teip Payment', 'teip-payment-gateway'),
            'description' => __('Pay securely using Teip.', 'teip-payment-gateway'),
        ];
    }

    public function get_configuration() {
        return [];
    }
}
